unit SDK.DeliveryPharmacy.V2;

interface

uses
  System.JSON,
  REST.Client,
  REST.Types;

type
  iDeliveryPharmacyResponse = interface
    function StatusCode: Integer;
    function StatusText: String;
    function JSON: TJSONValue;
    function isOK: Boolean;
  end;

  TDeliveryPharmacyResponse = class(TInterfacedObject, iDeliveryPharmacyResponse)
  private
    FStatusCode: Integer;
    FJSON: TJSONValue;
    FStatusText: String;
  public
    constructor Create(ACodeStatus: Integer; AStatusText: String; AJSON: TJSONValue);

    function StatusCode: Integer;
    function StatusText: String;
    function JSON: TJSONValue;
    function isOK: Boolean;
  end;

  TDeliveryPharmacy = class
  private
    FRESTRequest: TRESTRequest;
    FRESTClient: TRESTClient;
    FRESTResponse: TRESTResponse;

    FResponse: iDeliveryPharmacyResponse;

    class var FToken: String;
    class var FID_Empresa: String;
    class var FID_ERP: String;

    const
      _BASE_URL = 'https://api.deliverypharmacy.com.br';

    procedure Reset;
    procedure Execute(AMethod: TRESTRequestMethod; AResource: String; ABody: TJSONValue = nil);
  public
    constructor Create; virtual;
    destructor Destroy; override;

    class procedure Configure(AToken, AID_Empresa, AID_ERP: String);

    procedure PostProduto(ABody: TJSONObject);
    procedure PutProduto(AID: String; ABody: TJSONObject);
    procedure PutProdutoValor(AID: String; ABody: TJSONObject);

    procedure PostPromocao(ABody: TJSONObject);
    procedure PutPromocao(AID: String; ABody: TJSONObject);
    procedure DeletePromocao(AID: String);
    procedure PostPromocaoProduto(AIDPromocao, AIDProduto:String; ABody: TJSONObject);
    procedure DeletePromocaoProduto(AIDPromocao, AIDProduto: String);

    procedure PostFormaPagto(ABody: TJSONObject);

    procedure GetPedidos;
    procedure GetPedido(AID: String);
    procedure PutPedidoSincronizado(AID: String);
    procedure PutPedidoCancelado(AID, AMotivo: String);

    function Response: iDeliveryPharmacyResponse;
  end;

implementation

uses
  System.SysUtils;

{ TDeliveryPharmacy }

class procedure TDeliveryPharmacy.Configure(AToken, AID_Empresa, AID_ERP: String);
begin
  FToken := AToken;
  FID_Empresa := AID_Empresa;
  FID_ERP := AID_ERP;
end;

constructor TDeliveryPharmacy.Create;
begin
  inherited Create;

  FRESTRequest := TRESTRequest.Create(nil);
  FRESTClient := TRESTClient.Create(nil);
  FRESTResponse := TRESTResponse.Create(nil);

  FRESTRequest.Client := FRESTClient;
  FRESTRequest.Response := FRESTResponse;

  Reset;
end;

procedure TDeliveryPharmacy.DeletePromocao(AID: String);
begin
  Execute(TRESTRequestMethod.rmDELETE, 'v2/promocao/' + AID);
end;

procedure TDeliveryPharmacy.DeletePromocaoProduto(AIDPromocao, AIDProduto: String);
begin
  Execute(TRESTRequestMethod.rmDELETE, 'v2/promocao/' + AIDPromocao + '/produto/' + AIDProduto);
end;

destructor TDeliveryPharmacy.Destroy;
begin
  FRESTRequest.Free;
  FRESTClient.Free;
  FRESTResponse.Free;

  inherited;
end;

procedure TDeliveryPharmacy.Execute(AMethod: TRESTRequestMethod; AResource: String; ABody: TJSONValue = nil);
begin
  try
    Reset;
    FRESTRequest.Method := AMethod;

    if Assigned(ABody) then
      FRESTRequest.Body.Add(ABody.ToString, TRESTContentType.ctAPPLICATION_JSON);

    FRESTRequest.Params.AddItem('Authorization', FToken, TRESTRequestParameterKind.pkHTTPHEADER, [TRESTRequestParameterOption.poDoNotEncode]);
    FRESTRequest.Params.AddItem('x-id-erp', FID_ERP, TRESTRequestParameterKind.pkHTTPHEADER, [TRESTRequestParameterOption.poDoNotEncode]);
    FRESTRequest.Params.AddItem('x-id-empresa', FID_Empresa, TRESTRequestParameterKind.pkHTTPHEADER, [TRESTRequestParameterOption.poDoNotEncode]);

    FRESTRequest.Resource := AResource;

    try
      FRESTRequest.Execute;
    except
    end;

    FResponse := TDeliveryPharmacyResponse.Create(FRESTResponse.StatusCode, FRESTResponse.StatusText, FRESTResponse.JSONValue);
  finally
    if Assigned(ABody) then
      ABody.Free;
  end;
end;

procedure TDeliveryPharmacy.GetPedido(AID: String);
begin
  Execute(TRESTRequestMethod.rmGET, 'v2/pedido/' + AID);
end;

procedure TDeliveryPharmacy.GetPedidos;
begin
  Execute(TRESTRequestMethod.rmGET, 'v2/pedido');
end;

procedure TDeliveryPharmacy.PostFormaPagto(ABody: TJSONObject);
begin
  Execute(TRESTRequestMethod.rmPOST, 'v2/formapagto', ABody);
end;

procedure TDeliveryPharmacy.PostProduto(ABody: TJSONObject);
begin
  Execute(TRESTRequestMethod.rmPOST, 'v2/produto', ABody);
end;

procedure TDeliveryPharmacy.PostPromocao(ABody: TJSONObject);
begin
  Execute(TRESTRequestMethod.rmPOST, 'v2/promocao', ABody);
end;

procedure TDeliveryPharmacy.PostPromocaoProduto(AIDPromocao, AIDProduto:String; ABody: TJSONObject);
begin
  Execute(TRESTRequestMethod.rmPOST, 'v2/promocao/' + AIDPromocao + '/produto/' + AIDProduto, ABody);
end;

procedure TDeliveryPharmacy.PutPedidoCancelado(AID, AMotivo: String);
begin
  Execute(TRESTRequestMethod.rmPUT, 'v2/pedido/' + AID + '/cancelado', TJSONObject.Create.AddPair('motivo', AMotivo));
end;

procedure TDeliveryPharmacy.PutPedidoSincronizado(AID: String);
begin
  Execute(TRESTRequestMethod.rmPUT, 'v2/pedido/' + AID + '/sincronizado');
end;

procedure TDeliveryPharmacy.PutProduto(AID: String; ABody: TJSONObject);
begin
  Execute(TRESTRequestMethod.rmPUT, 'v2/produto/' + AID, ABody);
end;

procedure TDeliveryPharmacy.PutProdutoValor(AID: String; ABody: TJSONObject);
begin
  Execute(TRESTRequestMethod.rmPUT, 'v2/produto/' + AID + '/valor', ABody);
end;

procedure TDeliveryPharmacy.PutPromocao(AID: String; ABody: TJSONObject);
begin
  Execute(TRESTRequestMethod.rmPUT, 'v2/promocao/' + AID, ABody);
end;

procedure TDeliveryPharmacy.Reset;
begin
  FRESTRequest.ResetToDefaults;
  FRESTClient.ResetToDefaults;
  FRESTResponse.ResetToDefaults;

  FRESTRequest.Params.Clear;
  FRESTRequest.Resource := EmptyStr;
  FRESTRequest.AcceptEncoding := 'gzip';
  FRESTRequest.AutoCreateParams := True;

  FRESTClient.BaseURL := _BASE_URL;
  FRESTClient.FallbackCharsetEncoding := 'UTF-8';
end;

function TDeliveryPharmacy.Response: iDeliveryPharmacyResponse;
begin
  Result := FResponse;
end;

{ TDeliveryPharmacyResponse }

function TDeliveryPharmacyResponse.StatusCode: Integer;
begin
  Result := FStatusCode;
end;

constructor TDeliveryPharmacyResponse.Create(ACodeStatus: Integer; AStatusText: String; AJSON: TJSONValue);
begin
  FStatusCode := ACodeStatus;
  FJSON := AJSON;
  FStatusText := AStatusText;
end;

function TDeliveryPharmacyResponse.isOK: Boolean;
begin
  Result := (FStatusCode >= 200) and (FStatusCode <= 299);
end;

function TDeliveryPharmacyResponse.JSON: TJSONValue;
begin
  Result := FJSON;
end;

function TDeliveryPharmacyResponse.StatusText: String;
begin
  Result := FStatusText;
end;

end.
